/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.sis.referencing.operation.provider;

import jakarta.xml.bind.annotation.XmlTransient;
import org.opengis.util.FactoryException;
import org.opengis.parameter.ParameterDescriptor;
import org.opengis.parameter.ParameterDescriptorGroup;
import org.opengis.parameter.ParameterNotFoundException;
import org.opengis.referencing.cs.EllipsoidalCS;
import org.opengis.referencing.operation.Conversion;
import org.opengis.referencing.operation.MathTransform;
import org.apache.sis.measure.Units;
import org.apache.sis.measure.Longitude;
import org.apache.sis.parameter.Parameters;
import org.apache.sis.parameter.ParameterBuilder;
import org.apache.sis.referencing.operation.projection.ZonedGridSystem;


/**
 * The provider for <q>Transverse Mercator Zoned Grid System</q> projection (EPSG:9824).
 *
 * @author  Martin Desruisseaux (Geomatys)
 */
@XmlTransient
public final class ZonedTransverseMercator extends AbstractProvider {
    /**
     * For cross-version compatibility.
     */
    private static final long serialVersionUID = 4555131921419380461L;

    /**
     * The operation parameter descriptor for the <cite>Initial longitude</cite> (λ₁) parameter value.
     *
     * <!-- Generated by ParameterNameTableGenerator -->
     * <table class="sis">
     *   <caption>Parameter names</caption>
     *   <tr><td> EPSG:    </td><td> Initial longitude </td></tr>
     * </table>
     * <b>Notes:</b>
     * <ul>
     *   <li>Default value: -180°</li>
     * </ul>
     */
    public static final ParameterDescriptor<Double> INITIAL_LONGITUDE;

    /**
     * The operation parameter descriptor for the <cite>Zone width</cite> (W) parameter value.
     *
     * <!-- Generated by ParameterNameTableGenerator -->
     * <table class="sis">
     *   <caption>Parameter names</caption>
     *   <tr><td> EPSG:    </td><td> Zone width </td></tr>
     * </table>
     * <b>Notes:</b>
     * <ul>
     *   <li>Value domain: (0.0 … ∞)°</li>
     *   <li>Default value: 6°</li>
     * </ul>
     */
    public static final ParameterDescriptor<Double> ZONE_WIDTH;

    /**
     * The group of all parameters expected by this coordinate operation.
     */
    private static final ParameterDescriptorGroup PARAMETERS;
    static {
        final ParameterBuilder builder = builder();
        INITIAL_LONGITUDE = builder.addIdentifier("8830").addName("Initial longitude")
                .createBounded(Longitude.MIN_VALUE, Longitude.MAX_VALUE, Longitude.MIN_VALUE, Units.DEGREE);

        ZONE_WIDTH = builder.addIdentifier("8831").addName("Zone width")
                .createStrictlyPositive(6, Units.DEGREE);

        PARAMETERS = builder
                .addIdentifier("9824")
                .addName("Transverse Mercator Zoned Grid System")
                .addName("UTM grid system")
                .addName("UTM")
                .createGroupForMapProjection(
                        TransverseMercator.LATITUDE_OF_ORIGIN,
                        INITIAL_LONGITUDE,
                        ZONE_WIDTH,
                        TransverseMercator.SCALE_FACTOR,
                        TransverseMercator.FALSE_EASTING,
                        TransverseMercator.FALSE_NORTHING);
    }

    /**
     * Constructs a new provider.
     * We do not classify this operation as a cylindrical projection
     * because of the discontinuities between zones.
     */
    public ZonedTransverseMercator() {
        super(Conversion.class, PARAMETERS,
              EllipsoidalCS.class, true,
              EllipsoidalCS.class, false,
              (byte) 2);
    }

    /**
     * Creates a map projection from the specified group of parameter values.
     *
     * @param  context  the parameter values together with its context.
     * @return the map projection created from the given parameter values.
     * @throws ParameterNotFoundException if a required parameter was not found.
     * @throws FactoryException if the map projection cannot be created.
     */
    @Override
    public final MathTransform createMathTransform(final Context context) throws FactoryException {
        return new ZonedGridSystem(this, Parameters.castOrWrap(context.getCompletedParameters()), context.getFactory());
    }
}
