/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.elasticjob.reg.zookeeper;

import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.RetryOneTime;
import org.apache.shardingsphere.elasticjob.reg.zookeeper.env.RegistryCenterEnvironmentPreparer;
import org.apache.shardingsphere.elasticjob.test.util.EmbedTestingServer;
import org.apache.zookeeper.KeeperException.NoAuthException;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;

class ZookeeperRegistryCenterForAuthTest {
    
    private static final EmbedTestingServer EMBED_TESTING_SERVER = new EmbedTestingServer();
    
    private static final String NAME_SPACE = ZookeeperRegistryCenterForAuthTest.class.getName();
    
    private static ZookeeperRegistryCenter zkRegCenter;
    
    @BeforeAll
    static void setUp() {
        EMBED_TESTING_SERVER.start();
        ZookeeperConfiguration zookeeperConfiguration = new ZookeeperConfiguration(EMBED_TESTING_SERVER.getConnectionString(), NAME_SPACE);
        zookeeperConfiguration.setDigest("digest:password");
        zookeeperConfiguration.setSessionTimeoutMilliseconds(5000);
        zookeeperConfiguration.setConnectionTimeoutMilliseconds(5000);
        zkRegCenter = new ZookeeperRegistryCenter(zookeeperConfiguration);
        zkRegCenter.init();
        RegistryCenterEnvironmentPreparer.persist(zkRegCenter);
    }
    
    @AfterAll
    static void tearDown() {
        zkRegCenter.close();
    }
    
    @Test
    void assertInitWithDigestSuccess() throws Exception {
        try (
                CuratorFramework client = CuratorFrameworkFactory.builder()
                        .connectString(EMBED_TESTING_SERVER.getConnectionString())
                        .retryPolicy(new RetryOneTime(2000))
                        .authorization("digest", "digest:password".getBytes()).build()) {
            client.start();
            client.blockUntilConnected();
            assertThat(client.getData().forPath("/" + ZookeeperRegistryCenterForAuthTest.class.getName() + "/test/deep/nested"), is("deepNested".getBytes()));
        }
    }
    
    @Test
    void assertInitWithDigestFailure() {
        assertThrows(NoAuthException.class, () -> {
            try (CuratorFramework client = CuratorFrameworkFactory.newClient(EMBED_TESTING_SERVER.getConnectionString(), new RetryOneTime(2000))) {
                client.start();
                client.blockUntilConnected();
                client.getData().forPath("/" + ZookeeperRegistryCenterForAuthTest.class.getName() + "/test/deep/nested");
            }
        });
    }
}
