/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.iotdb.db.queryengine.plan.analyze.cache.schema;

import org.apache.iotdb.commons.service.metric.enums.Metric;
import org.apache.iotdb.commons.service.metric.enums.Tag;
import org.apache.iotdb.metrics.AbstractMetricService;
import org.apache.iotdb.metrics.metricsets.IMetricSet;
import org.apache.iotdb.metrics.utils.MetricLevel;
import org.apache.iotdb.metrics.utils.MetricType;

import java.util.Objects;

public class DataNodeSchemaCacheMetrics implements IMetricSet {

  private static final String SCHEMA_CACHE_TOTAL_USAGE = "schema_cache_total_usage";
  private static final String SCHEMA_CACHE_MEM_CAPACITY = "schema_cache_mem_capacity";

  private final DeviceSchemaCache deviceSchemaCache;

  public DataNodeSchemaCacheMetrics(final DeviceSchemaCache dataNodeSchemaCache) {
    this.deviceSchemaCache = dataNodeSchemaCache;
  }

  @Override
  public void bindTo(final AbstractMetricService metricService) {
    metricService.createAutoGauge(
        Metric.CACHE.toString(),
        MetricLevel.IMPORTANT,
        deviceSchemaCache,
        DeviceSchemaCache::getHitCount,
        Tag.NAME.toString(),
        "SchemaCache",
        Tag.TYPE.toString(),
        "hit");
    metricService.createAutoGauge(
        Metric.CACHE.toString(),
        MetricLevel.IMPORTANT,
        deviceSchemaCache,
        DeviceSchemaCache::getRequestCount,
        Tag.NAME.toString(),
        "SchemaCache",
        Tag.TYPE.toString(),
        "all");
    metricService.createAutoGauge(
        Metric.MEM.toString(),
        MetricLevel.IMPORTANT,
        deviceSchemaCache,
        DeviceSchemaCache::getMemoryUsage,
        Tag.NAME.toString(),
        SCHEMA_CACHE_TOTAL_USAGE);
    metricService.createAutoGauge(
        Metric.MEM.toString(),
        MetricLevel.IMPORTANT,
        deviceSchemaCache,
        DeviceSchemaCache::capacity,
        Tag.NAME.toString(),
        SCHEMA_CACHE_MEM_CAPACITY);
    metricService.createAutoGauge(
        Metric.CACHE_ENTRIES_NUM.toString(),
        MetricLevel.IMPORTANT,
        deviceSchemaCache,
        DeviceSchemaCache::entriesCount);
  }

  @Override
  public void unbindFrom(final AbstractMetricService metricService) {
    metricService.remove(
        MetricType.AUTO_GAUGE,
        Metric.CACHE.toString(),
        Tag.NAME.toString(),
        "SchemaCache",
        Tag.TYPE.toString(),
        "hit");
    metricService.remove(
        MetricType.AUTO_GAUGE,
        Metric.CACHE.toString(),
        Tag.NAME.toString(),
        "SchemaCache",
        Tag.TYPE.toString(),
        "all");
    metricService.remove(
        MetricType.AUTO_GAUGE,
        Metric.MEM.toString(),
        Tag.NAME.toString(),
        SCHEMA_CACHE_TOTAL_USAGE);
    metricService.remove(
        MetricType.AUTO_GAUGE,
        Metric.MEM.toString(),
        Tag.NAME.toString(),
        SCHEMA_CACHE_MEM_CAPACITY);
    metricService.remove(MetricType.AUTO_GAUGE, Metric.CACHE_ENTRIES_NUM.toString());
  }

  @Override
  public boolean equals(final Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    final DataNodeSchemaCacheMetrics that = (DataNodeSchemaCacheMetrics) o;
    return Objects.equals(deviceSchemaCache, that.deviceSchemaCache);
  }

  @Override
  public int hashCode() {
    return Objects.hash(deviceSchemaCache);
  }
}
